/*
     // - - - - - - - - - - - -	- //
    //   Random Race Generator   //
   // - - - - - - - - - - - - - // 

	Description:
	 This script allows the player to randomly 
	 create races across the San Andreas map.
	 
	 While doing this, it provides a neat GUI for 
	 viewing the current available races.
	 
	Commands:
	 /joinrace
		- opens the race map where you can create
		  new races or join current ones.
	 /startrace		
		- starts the current race.
		  (only works for the creator of the race)
	 /leaverace
		- (if creator) destroys the current race.
		  (if contestant) leaves the current race.
	 /respawninrace
		- respawns the player at the last checkpoint.
		
	Contest:
	 This script was made for the RouteConnector
	 contest, hosted by Gamer_Z.
	 
	 URL:
	 http://forum.sa-mp.com/showthread.php?t=411412
	
	Thanks a lot:
	 - Mauzen;
	 - Gamer_Z;
	 - All the people who contributed without them 
	   actually knowing they contributed.
	
	If you have any problems with or questions
	 about this script, please contact me via the 
	 official SA-MP forums.
	 Topic URL:
	 http://forum.sa-mp.com/showthread.php?t=437708
	
	Regards,
	 Basssiiie
	 
*/

// ---------------------------------------------
// --- > --- > --- MAIN SETTINGS --- < --- < ---
// ---------------------------------------------


// This is the maximum amount of races shown in the list.
// Note: If this number is too high, it will overlap the "Create", "Join" and "Close" buttons.
// Default: 14
#define MAX_RACES 14


// This is the "maximum" distance a race can be in meters.
// Note: The race won't stop at this exact number, it will try to find a finish point as quickly as possible.
// Default: 30000
#define MAX_RACE_DISTANCE 30000


// This defines the minimal distance a race can be in meters.
// Default: 500
#define MIN_RACE_DISTANCE 500


// This number defines the maximum amount of checkpoints that will be saved into memory.
// Note 1: If a race exceeds this number, it will immediately spawn a finish point at the last checkpoint.
// Note 2: Try to find a good balance between this number and MAX_RACE_DISTANCE.
// Default: 250
#define MAX_CHECKPOINTS 250


// This number is the maximum amount of textdraw icons that will be shown on the /joinrace map.
// Default: 100
#define MAX_TEXTDRAW_ICONS 100


// This is the maximum amount of contestans that are allowed to join a race.
// Note 1: This is including the race starter! 
// Note 2: The higher the number, the more vehicles have to be spawned, the higher the chance that they'll get stuck inside buildings.
// Default: 8
#define MAX_CONTESTANTS 8


// This is the minimum distance between checkpoints. 
// Default: 100.0
#define MINIMAL_DISTANCE_CP 100.0


// This defines whether to remember the old position of a player, before he joins a race.
// Note: When set to true, the player will be spawned at his old position after the race finishes.
// Default: false
#define REMEMBER_OLD_POSITION false



// ------------------------------------------------
// --- > --- > --- VEHICLE SETTINGS --- < --- < ---
// ------------------------------------------------


// This defines the vehicle model to be used in the races.
// Default: 415
#define RACE_VEHICLE_MODEL 415


// These values define the vehicle colors used on the model defined at RACE_VEHICLE_MODEL.
// Default: -1 and -1
#define RACE_VEHICLE_COL1 -1
#define RACE_VEHICLE_COL2 -1



// --------------------------------------------------
// --- > --- > --- TECHNICAL SETTINGS --- < --- < ---
// --------------------------------------------------


// This is the PVar-tag that will be used for Player Variables.
// Note: This is used to prevent conflicts with other scripts and their variables.
// Default: "RRG_"
#define PVAR_TAG "RRG_"


// This number is the offset of the ID which will be used for dialogs.
// Note: Use an unique number which doesn't come close to other IDs used in other scripts.
// Default: 1357
#define DIALOG_OFFSET 1357



// ----------------------------------------------
// --- > --- > --- COLOR SETTINGS --- < --- < ---
// ----------------------------------------------


// This color is used for empty race slots in the /joinrace menu.
// Default: 0xFFFFFFFF (White)
#define COL_MENU_REGULAR 0xFFFFFFFF


// This color is shown when you move your mouse over a race slot in the /joinrace menu.
// Default: 0xDD8080FF (Indian/light red)
#define COL_MENU_MOUSEOVER 0xDD8080FF


// This color is used when you select one of the slots in the /joinrace menu.
// Default: 0xCF2C23FF (Firebrick/dark red)
#define COL_MENU_SELECTED 0xCF2C23FF


// This color is used for a race slot which can't be joined anymore.
// Default: 0x5B0000FF (Very dark red)
#define COL_MENU_STARTED 0x5B0000FF


// This is the color which is used for the regular chat text.
// Default: 0xFFFFFFFF (Very light red, almost white)
#define COL_TEXT_REG 0xFFFFFFFF


// This is the color which is used for the winners chat text.
// Default: 0xFF3E3EFF (Just red)
#define COL_TEXT_WIN 0xFF3E3EFF


// This is the color which is used for the chat errors.
// Default: 0xD21313FF (Firebrick/dark red)
#define COL_TEXT_ERROR 0xD21313FF


// This color is used for the (radar) map icons which suggest the next checkpoints.
// Default: 0x5B0000FF (Very dark red)
#define COL_MAP_CP 0x5B0000FF



// -----------------------------------------------
// --- > --- > --- END OF SETTINGS --- < --- < ---
// -----------------------------------------------



#include <a_samp>
#include <RouteConnector>


new raceOwners[MAX_RACES] = {INVALID_PLAYER_ID, ...};
new raceFinishedPeople[MAX_RACES char];
new bool: raceStarted[MAX_RACES char];
new raceEndTimers[MAX_RACES char] = {-1, ...};
new Float: raceDistance[MAX_RACES];
new Float: raceCheckpointList[MAX_RACES][MAX_CHECKPOINTS][3];
new Text: raceMapIcons[MAX_RACES][MAX_TEXTDRAW_ICONS];

new Text: joinMenuButtons[3];
new Text: joinMenuSlots[MAX_RACES];
new Text: joinMenuExtra[3];


public OnGameModeInit() return onScriptInit();
public OnFilterScriptInit() return onScriptInit();
public OnGameModeExit() return onScriptExit();
public OnFilterScriptExit() return onScriptExit();

stock onScriptInit()
{
	joinMenuExtra[0] = TextDrawCreate(115.0, 125.0, "_");
	TextDrawUseBox(joinMenuExtra[0], true);
	TextDrawLetterSize(joinMenuExtra[0], 1.0, 30.0);
	TextDrawBoxColor(joinMenuExtra[0], 0x77);
	TextDrawTextSize(joinMenuExtra[0], 530.0, 400.0);

	joinMenuExtra[1] = TextDrawCreate(270.0, 135.0, "samaps:map");
	//TextDrawUseBox(joinMenuExtra[1], true);
	TextDrawFont(joinMenuExtra[1], 4);
	TextDrawTextSize(joinMenuExtra[1], 250.0, 250.0);
	
	joinMenuExtra[2] = TextDrawCreate(128.0, 108.0, "Random Race Generator");
	TextDrawFont(joinMenuExtra[2], 0);
	TextDrawLetterSize(joinMenuExtra[2], 1.0, 2.75);
	TextDrawSetOutline(joinMenuExtra[2], 2);
	
	joinMenuButtons[0] = TextDrawCreate(125.0, 370.0, " Create");
	joinMenuButtons[1] = TextDrawCreate(125.0, 370.0, " Join");
	joinMenuButtons[2] = TextDrawCreate(195.0, 370.0, " Close");
	
	for (new b; b != sizeof(joinMenuButtons); b++)
	{
		TextDrawColor(joinMenuButtons[b], COL_MENU_REGULAR);
		TextDrawLetterSize(joinMenuButtons[b], 0.4, 1.5);
		TextDrawSetOutline(joinMenuButtons[b], 1);
		TextDrawUseBox(joinMenuButtons[b], true);
		TextDrawBoxColor(joinMenuButtons[b], 0x55);
		if (b == 2)
		{
			TextDrawTextSize(joinMenuButtons[b], 255.0, 12.0);
		}
		else
		{
			TextDrawTextSize(joinMenuButtons[b], 185.0, 12.0);
		}
		TextDrawSetSelectable(joinMenuButtons[b], true);
	}

	for (new s; s != MAX_RACES; s++)
	{
		joinMenuSlots[s] = TextDrawCreate(125.0, 140.0 + float(s * 15), "<Empty> Create a race!");
		TextDrawColor(joinMenuSlots[s], COL_MENU_REGULAR);
		TextDrawLetterSize(joinMenuSlots[s], 0.25, 1.2);
		TextDrawSetOutline(joinMenuSlots[s], 1);
		TextDrawTextSize(joinMenuSlots[s], 275.0, 12.0);
		TextDrawSetSelectable(joinMenuSlots[s], true);
		
		for (new i; i != MAX_TEXTDRAW_ICONS; i++)
		{
			raceMapIcons[s][i] = Text: INVALID_TEXT_DRAW;		
		}
	}
	return 1;
}

stock onScriptExit()
{
	for (new b; b != sizeof(joinMenuButtons); b++)
	{
		TextDrawHideForAll(joinMenuButtons[b]);
		TextDrawDestroy(joinMenuButtons[b]);
	}
	for (new e; e != sizeof(joinMenuExtra); e++)
	{
		TextDrawHideForAll(joinMenuExtra[e]);
		TextDrawDestroy(joinMenuExtra[e]);
	}
	for (new s; s != MAX_RACES; s++)
	{
		cleanRace(s);
		TextDrawHideForAll(joinMenuSlots[s]);
		TextDrawDestroy(joinMenuSlots[s]);
	}
	return 1;
}

public OnPlayerSpawn(playerid)
{
	new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
	if (race)
	{
		race -= 2;
		if (raceOwners[race] != INVALID_PLAYER_ID)
		{
			respawnPlayer(playerid, race);
		}
	}
	return 1;
}

public OnVehicleDeath(vehicleid, killerid)
{
	for (new p, mp = GetMaxPlayers(); p != mp; p++)
	{
		if (IsPlayerConnected(p) && !IsPlayerNPC(p))
		{
			new race = GetPVarInt(p, PVAR_TAG"currentRaceID");
			if (race)
			{
				race -= 2;
				if (raceOwners[race] != INVALID_PLAYER_ID)
				{
					if (GetPVarInt(p, PVAR_TAG"currentVehID") == vehicleid)
					{
						respawnPlayer(p, race);
					}
				}
			}
		}
	}
	return 1;
}

public OnPlayerDisconnect(playerid, reason)
{
	#if REMEMBER_OLD_POSITION == true
	removePlayerFromRace(playerid, false);
	#else
	removePlayerFromRace(playerid);
	#endif
	return 1;
}

public OnPlayerCommandText(playerid, cmdtext[])
{
	if (!strcmp(cmdtext, "/joinrace", true))
	{
		showJoinMenu(playerid);
		return 1;
	}
	
	if (!strcmp(cmdtext, "/startrace", true))
	{
		new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
		if (race && raceOwners[race - 2] == playerid)
		{
			startRace(race - 2);
		}
		else
		{
			if (race)
			{
				SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You have not started this race.");
			}
			else
			{
				SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You have not started a race yet.");
			}
		}
		return 1;
	}
	
	if (!strcmp(cmdtext, "/leaverace", true))
	{
		new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
		if (race && raceOwners[race - 2] == playerid)
		{
			SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You have called the race off.");
		}
		else
		{
			SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You have left the race.");
		}
		removePlayerFromRace(playerid);
		return 1;
	}
	
	if (!strcmp(cmdtext, "/respawninrace", true))
	{
		new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
		if (race)
		{
			race -= 2;
			if (raceStarted{race} == false)
			{
				return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You can't respawn right now! The race hasn't started yet.");
			}
			new cp = GetPVarInt(playerid, PVAR_TAG"currentCPID");
			if (!raceCheckpointList[race][cp][0] && !raceCheckpointList[race][cp][1] && !raceCheckpointList[race][cp][2])
			{
				return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You can't respawn anymore! You have already finished.");
			}
			respawnPlayer(playerid, race);
		}
		else
		{
			SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You are not om a race right now!");
		}
		return 1;	
	}
	return 0;
}

public OnPlayerClickTextDraw(playerid, Text:clickedid)
{
	new menuopen = GetPVarInt(playerid, PVAR_TAG"joinMenuOpen");
	if (menuopen)
	{
		menuopen -= 2;
		if (clickedid == Text: INVALID_TEXT_DRAW || clickedid == Text: joinMenuButtons[2])
		{
			hideJoinMenu(playerid);	
			return 1;
		}
		if (clickedid == Text: joinMenuButtons[0] && raceOwners[menuopen] == INVALID_PLAYER_ID)
		{
			CancelSelectTextDraw(playerid);
			createRace(playerid, menuopen);
			return 1;
		}
		if (clickedid == Text: joinMenuButtons[1] && raceOwners[menuopen] != INVALID_PLAYER_ID)
		{
			CancelSelectTextDraw(playerid);
			joinRace(playerid, menuopen);
			return 1;
		}
		for (new i; i != MAX_RACES; i++)
		{
			if (clickedid == joinMenuSlots[i])
			{
				SetPVarInt(playerid, PVAR_TAG"joinMenuOpen", i + 2);
				TextDrawColor(joinMenuSlots[i], COL_MENU_SELECTED);
				TextDrawShowForPlayer(playerid, joinMenuSlots[i]);
				
				if (raceOwners[i] == INVALID_PLAYER_ID)
				{
					TextDrawShowForPlayer(playerid, joinMenuButtons[0]);
					TextDrawHideForPlayer(playerid, joinMenuButtons[1]);
				}
				else
				{
					TextDrawHideForPlayer(playerid, joinMenuButtons[0]);
					TextDrawShowForPlayer(playerid, joinMenuButtons[1]);
					
					for (new c; c != MAX_TEXTDRAW_ICONS; c++)
					{
						if (raceMapIcons[i][c] != Text: INVALID_TEXT_DRAW)
						{
							TextDrawShowForPlayer(playerid, raceMapIcons[i][c]);
						}
					}
				}
			}
			else
			{
				if (raceStarted{i} == true)
				{
					TextDrawColor(joinMenuSlots[i], COL_MENU_STARTED);
				}
				else
				{
					TextDrawColor(joinMenuSlots[i], COL_MENU_REGULAR);
				}
				TextDrawShowForPlayer(playerid, joinMenuSlots[i]);
				
				for (new c; c != MAX_TEXTDRAW_ICONS; c++)
				{
					if (raceMapIcons[i][c] != Text: INVALID_TEXT_DRAW)
					{
						TextDrawHideForPlayer(playerid, raceMapIcons[i][c]);
					}
				}
			}
		}
	}
	return 0;
}

public OnDialogResponse(playerid, dialogid, response, listitem, inputtext[])
{
	switch (dialogid)
	{
		case DIALOG_OFFSET:
		{
			if (!response) 
			{
				DeletePVar(playerid, PVAR_TAG"closestNode");
				DeletePVar(playerid, PVAR_TAG"currentRaceID");
				return 1;
			}
			new closestNode = GetPVarInt(playerid, PVAR_TAG"closestNode");
			if (!closestNode) return 1;
			
			new value = strval(inputtext);
			if (MIN_RACE_DISTANCE > value || value > MAX_RACE_DISTANCE)
			{
				return ShowPlayerDialog(playerid, DIALOG_OFFSET, 1, "Maximum length", "Please input the maximum length of the race in meters.\n\nERROR: The number must be between "#MIN_RACE_DISTANCE" and "#MAX_RACE_DISTANCE"!", "Generate", "Cancel");		
			}
			
			new slot = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
			if (slot && raceOwners[slot - 2] != INVALID_PLAYER_ID)
			{
				new freeRace = -1;
				for (new r; r != MAX_RACES; r++)
				{
					if (raceOwners[r] == INVALID_PLAYER_ID)
					{
						freeRace = r;
						break;
					}			
				}
				if (freeRace == -1)
				{
					SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: It's not possible to create more races at the moment!");
					DeletePVar(playerid, PVAR_TAG"closestNode");
					DeletePVar(playerid, PVAR_TAG"currentRaceID");
					return 1;
				}
				slot = freeRace + 2;
				SetPVarInt(playerid, PVAR_TAG"currentRaceID", slot);
			}
			
			SetPVarFloat(playerid, PVAR_TAG"totalRaceDistance", float(value));
			
			new anotherNode = NearestNodeFromPoint(float(random(6000) - 3000), float(random(6000) - 3000), 25.0, .IgnoreNodeID = closestNode);
			raceOwners[slot - 2] = playerid;
			CalculatePath(closestNode, anotherNode, slot - 2, .GrabNodePositions = true);
			return 1;
		}
	}
	return 0;
}

public GPS_WhenRouteIsCalculated(routeid, node_id_array[], amount_of_nodes, Float:distance, Float:Polygon[], Polygon_Size, Float:NodePosX[], Float:NodePosY[], Float:NodePosZ[])
{
	if (0 <= routeid < MAX_RACES && raceOwners[routeid] != INVALID_PLAYER_ID)
	{
		if (!amount_of_nodes || distance == -1) return cleanRace(routeid);
		new playerid = raceOwners[routeid];
		
		new lastIntersection, Float: distIntersection, curCPSlot = getFirstEmptyCPSlot(routeid), Float: maxDistance = GetPVarFloat(playerid, PVAR_TAG"totalRaceDistance"), i = 1, lastNode;
		
		if (curCPSlot == 0)
		{
			raceCheckpointList[routeid][0][0] = NodePosX[0];
			raceCheckpointList[routeid][0][1] = NodePosY[0];
			raceCheckpointList[routeid][0][2] = NodePosZ[0];
			raceCheckpointList[routeid][1][0] = NodePosX[1];
			raceCheckpointList[routeid][1][1] = NodePosY[1];
			raceCheckpointList[routeid][1][2] = NodePosZ[1];
			
			curCPSlot += 2;
		}
		for (; i != amount_of_nodes; i++)
		{
			new Float: dist = GetDistanceBetweenNodes(node_id_array[i - 1], node_id_array[i]);
			distIntersection += dist;
			raceDistance[routeid] += dist;
			
			if (IsNodeIntersection(node_id_array[i]) == 1)
			{
				if (distIntersection > MINIMAL_DISTANCE_CP / 2)
				{
					new Float: averageDist = distIntersection / float(floatround(distIntersection / MINIMAL_DISTANCE_CP, floatround_floor)), Float: curDist, bool: limitreached;
					
					for(new n = lastIntersection + 1; n != i; n++)
					{
						curDist += GetDistanceBetweenNodes(node_id_array[n - 1], node_id_array[n]);
						if (curDist > averageDist)
						{
							raceCheckpointList[routeid][curCPSlot][0] = NodePosX[n];
							raceCheckpointList[routeid][curCPSlot][1] = NodePosY[n];
							raceCheckpointList[routeid][curCPSlot][2] = NodePosZ[n];
							curCPSlot++;
							curDist = 0.0;
							
							if (curCPSlot == MAX_CHECKPOINTS)
							{
								limitreached = true;
								break;
							}
						}
					}
					
					if (limitreached == false)
					{
						raceCheckpointList[routeid][curCPSlot][0] = NodePosX[i];
						raceCheckpointList[routeid][curCPSlot][1] = NodePosY[i];
						raceCheckpointList[routeid][curCPSlot][2] = NodePosZ[i];
						curCPSlot++;
						lastNode = node_id_array[i];
				
						lastIntersection = i;
						distIntersection = 0.0;
					}
									
					if (raceDistance[routeid] >= maxDistance || curCPSlot == MAX_CHECKPOINTS)
					{
						new str[48], pName[MAX_PLAYER_NAME], len;
						len = GetPlayerName(playerid, pName, MAX_PLAYER_NAME);
						switch (pName[len - 1])
						{
							case 's', 'z': format(str, sizeof(str), "%s' race (%.1fm)", pName, raceDistance[routeid]);
							default: format(str, sizeof(str), "%s's race (%.1fm)", pName, raceDistance[routeid]);
						}
						TextDrawSetString(joinMenuSlots[routeid], str);
						
						new interval = (curCPSlot >= MAX_TEXTDRAW_ICONS) ? floatround(float(curCPSlot) / float(MAX_TEXTDRAW_ICONS), floatround_ceil) : 1;
						
						for (new c = 1, cp = 2; cp < curCPSlot && cp + interval <= MAX_CHECKPOINTS && c != MAX_TEXTDRAW_ICONS - 2; cp += interval)
						{
							if ((!raceCheckpointList[routeid][cp][0] && !raceCheckpointList[routeid][cp][1])) continue;
							
							raceMapIcons[routeid][c] = TextDrawCreate(267.0 + ((raceCheckpointList[routeid][cp][0] + 3000.0) / 6000.0) * 250.0, 382.0 - ((raceCheckpointList[routeid][cp][1] + 3000.0) / 6000.0) * 250.0, "hud:radar_light");
							TextDrawFont(raceMapIcons[routeid][c], 4);
							TextDrawTextSize(raceMapIcons[routeid][c], 6.0, 6.0);
							c++;
						}
						
						raceMapIcons[routeid][0] = TextDrawCreate(265.0 + ((raceCheckpointList[routeid][0][0] + 3000.0) / 6000.0) * 250.0, 380.0 - ((raceCheckpointList[routeid][0][1] + 3000.0) / 6000.0) * 250.0, "hud:radar_impound");
						raceMapIcons[routeid][MAX_TEXTDRAW_ICONS - 1] = TextDrawCreate(265.0 + ((raceCheckpointList[routeid][curCPSlot - 1][0] + 3000.0) / 6000.0) * 250.0, 380.0 - ((raceCheckpointList[routeid][curCPSlot - 1][1] + 3000.0) / 6000.0) * 250.0, "hud:radar_Flag");

						TextDrawFont(raceMapIcons[routeid][0], 4);
						TextDrawTextSize(raceMapIcons[routeid][0], 10.0, 10.0);
						TextDrawFont(raceMapIcons[routeid][MAX_TEXTDRAW_ICONS - 1], 4);
						TextDrawTextSize(raceMapIcons[routeid][MAX_TEXTDRAW_ICONS - 1], 10.0, 10.0);
						
						spawnInRace(raceOwners[routeid], routeid, 0);
						
						new text[128];
						format(text, sizeof(text), " [!] NOTE: %s has started a new race with a length of %.1f meters! Use /joinrace to join this race.", pName, raceDistance[routeid]);
						SendClientMessageToAll(COL_TEXT_REG, text);
						SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You can use /startrace to start the countdown or /leaverace to call the race off.");
						return 1;
					}
				}
			}
		}
		if (raceDistance[routeid] < maxDistance)
		{
			new newNode = NearestNodeFromPoint(float(random(6000) - 3000), float(random(6000) - 3000), 25.0, .IgnoreNodeID = lastNode);
			CalculatePath(lastNode, newNode, routeid, .GrabNodePositions = true);
		}
	}
	return 1;
}

stock Float:GetAngleToPos(Float: PX, Float: PY, Float:X, Float:Y)
{
	new Float:Angle = floatabs(atan((Y-PY)/(X-PX)));
	Angle = (X<=PX && Y>=PY) ? floatsub(180.0, Angle) : (X<PX && Y<PY) ? floatadd(Angle, 180.0) : (X>=PX && Y<=PY) ? floatsub(360.0, Angle) : Angle;
	Angle = floatsub(Angle, 90.0);
	Angle = (Angle >= 360.0) ? floatsub(Angle, 360.0) : Angle;
	Angle = (Angle <= 0.0) ? floatadd(Angle, 360.0) : Angle;
	return Angle;
}


stock createRace(playerid, slot)
{
	for (new r; r != MAX_RACES; r++)
	{
		if (raceOwners[r] == playerid)
		{
			return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You have already started a race!");
		}
	}
	
	new oldrace = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
	if (oldrace)
	{
		return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You are already in a race! Use /leaverace to leave that race.");
	}
	
	new closestNode = NearestPlayerNode(playerid, 50.0, 1);
	if (closestNode == -1)
	{
		return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You need to move closer to a road!");
	}
	SetPVarInt(playerid, PVAR_TAG"closestNode", closestNode);
	SetPVarInt(playerid, PVAR_TAG"currentRaceID", slot + 2);

	ShowPlayerDialog(playerid, DIALOG_OFFSET, 1, "Maximum length", "Please input the maximum length of the race in meters.", "Generate", "Cancel");
	return 1;
}

stock cleanRace(race, checkowner = false)
{
	new playerid = raceOwners[race];
	DeletePVar(playerid, PVAR_TAG"closestNode");
	DeletePVar(playerid, PVAR_TAG"totalRaceDistance");
	
	raceOwners[race] = INVALID_PLAYER_ID;
	raceDistance[race] = 0.0;
	raceFinishedPeople{race} = 0;
	raceStarted{race} = false;
	if (raceEndTimers{race} >= 0)
	{
		KillTimer(raceEndTimers{race});
	}
	raceEndTimers{race} = -1;
	
	for (new c; c != MAX_CHECKPOINTS; c++)
	{
		raceCheckpointList[race][c][0] = 0.0;
		raceCheckpointList[race][c][1] = 0.0;
		raceCheckpointList[race][c][2] = 0.0;
	}
	for (new i; i != MAX_TEXTDRAW_ICONS; i++)
	{
		TextDrawHideForAll(raceMapIcons[race][i]);
		TextDrawDestroy(raceMapIcons[race][i]);
		raceMapIcons[race][i] = Text: INVALID_TEXT_DRAW;
	}
	for (new p, mp = GetMaxPlayers(); p != mp; p++)
	{
		if (IsPlayerConnected(p) && !IsPlayerNPC(p))
		{
			new pRace = GetPVarInt(p, PVAR_TAG"currentRaceID");
			if (pRace && pRace - 2 == race)
			{
				if (!checkowner || (checkowner && p != playerid))
				{
					removePlayerFromRace(p);
				}
			}
		}
	}
	
	TextDrawSetString(joinMenuSlots[race], "<Empty> Create a race!");
	return 1;
}
#if REMEMBER_OLD_POSITION == true
stock removePlayerFromRace(playerid, bool: oldspawn = true)
#else
stock removePlayerFromRace(playerid)
#endif
{
	new veh = GetPVarInt(playerid, PVAR_TAG"currentVehID");
	DestroyVehicle(veh);
	DeletePVar(playerid, PVAR_TAG"currentVehID");
	DisablePlayerRaceCheckpoint(playerid);
	DeletePVar(playerid, PVAR_TAG"currentCPID");
	new timer = GetPVarInt(playerid, PVAR_TAG"respawnTimer");
	if (timer)
	{
		KillTimer(timer);
	}
	DeletePVar(playerid, PVAR_TAG"respawnTimer");
	
	for (new ci, icon = 99; ci != 2; ci++, icon--)
	{
		RemovePlayerMapIcon(playerid, icon);
	}
	
	#if REMEMBER_OLD_POSITION == true
	if (oldspawn == true)
	{
		new Float: oldpos[4], oldint;
		oldpos[0] = GetPVarFloat(playerid, PVAR_TAG"oldX");
		oldpos[1] = GetPVarFloat(playerid, PVAR_TAG"oldY");
		oldpos[2] = GetPVarFloat(playerid, PVAR_TAG"oldZ");
		oldpos[3] = GetPVarFloat(playerid, PVAR_TAG"oldR");
		oldint = GetPVarInt(playerid, PVAR_TAG"oldInt");
		SetPlayerPos(playerid, oldpos[0], oldpos[1], oldpos[2]);
		SetPlayerFacingAngle(playerid, oldpos[3]);
		SetPlayerInterior(playerid, oldint);
	}
	DeletePVar(playerid, PVAR_TAG"oldX");
	DeletePVar(playerid, PVAR_TAG"oldY");
	DeletePVar(playerid, PVAR_TAG"oldZ");
	DeletePVar(playerid, PVAR_TAG"oldR");
	DeletePVar(playerid, PVAR_TAG"oldInt");
	#endif
	TogglePlayerControllable(playerid, true);
	
	new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
	if (race)
	{
		race -= 2;
		for (new r; r != MAX_RACES; r++)
		{
			if (raceOwners[r] == playerid)
			{
				cleanRace(r, true);
			}
		}
	}
	DeletePVar(playerid, PVAR_TAG"currentRaceID");
	return 1;
}

stock showJoinMenu(playerid)
{
	SetPVarInt(playerid, PVAR_TAG"joinMenuOpen", 1);
	for (new e; e != sizeof(joinMenuExtra); e++)
	{
		TextDrawShowForPlayer(playerid, joinMenuExtra[e]);
	}
	for (new s; s != MAX_RACES; s++)
	{
		if (raceStarted{s} == true)
		{
			TextDrawColor(joinMenuSlots[s], COL_MENU_STARTED);
		}
		else
		{
			TextDrawColor(joinMenuSlots[s], COL_MENU_REGULAR);
		}
		TextDrawShowForPlayer(playerid, joinMenuSlots[s]);
	}
	TextDrawShowForPlayer(playerid, joinMenuButtons[2]);
	SelectTextDraw(playerid, COL_MENU_MOUSEOVER);
	return 1;
}

stock hideJoinMenu(playerid)
{
	CancelSelectTextDraw(playerid);
	DeletePVar(playerid, PVAR_TAG"joinMenuOpen");
	for (new b; b != sizeof(joinMenuButtons); b++)
	{
		TextDrawHideForPlayer(playerid, joinMenuButtons[b]);
	}
	for (new e; e != sizeof(joinMenuExtra); e++)
	{
		TextDrawHideForPlayer(playerid, joinMenuExtra[e]);
	}
	for (new s; s != MAX_RACES; s++)
	{
		TextDrawHideForPlayer(playerid, joinMenuSlots[s]);
		
		for (new i; i != MAX_TEXTDRAW_ICONS; i++)
		{
			TextDrawHideForPlayer(playerid, raceMapIcons[s][i]);
		}
	}
	return 1;
}

stock joinRace(playerid, race)
{
	new oldrace = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
	if (oldrace)
	{
		if (oldrace - 2 == race)
		{
			return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You are already in this race!");
		}
		else
		{
			return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You are already in another race! Use /leaverace to leave that race.");
		}
	}
	
	for (new r; r != MAX_RACES; r++)
	{
		if (raceOwners[r] == playerid)
		{
			return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You have already started another race! You can't join this one.");
		}
	}
	
	if (raceStarted{race} == true)
	{
		return SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: This race has already started! You can't join anymore.");	
	}
	
	new currentcontestants;
	for (new p, mp = GetMaxPlayers(); p != mp; p++)
	{
		if (IsPlayerConnected(p) && !IsPlayerNPC(p))
		{
			new pRace = GetPVarInt(p, PVAR_TAG"currentRaceID");
			if (pRace && pRace - 2 == race)
			{
				currentcontestants++;
			}
		}
	}
	if (currentcontestants > MAX_CONTESTANTS)
	{
		SendClientMessage(playerid, COL_TEXT_ERROR, " [!] ERROR: You can't join this race anymore! There's no room.");
		return 1;
	}
	SetPVarInt(playerid, PVAR_TAG"currentRaceID", race + 2);
	spawnInRace(playerid, race, currentcontestants);
	SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You can use /leaverace to leave this race.");
	return 1;
}

stock spawnInRace(playerid, race, spot = -1)
{
	#if REMEMBER_OLD_POSITION == true
	new Float: oldpos[4], oldint;
	GetPlayerPos(playerid, oldpos[0], oldpos[1], oldpos[2]);
	GetPlayerFacingAngle(playerid, oldpos[3]);
	oldint = GetPlayerInterior(playerid);
	SetPVarFloat(playerid, PVAR_TAG"oldX", oldpos[0]);
	SetPVarFloat(playerid, PVAR_TAG"oldY", oldpos[1]);
	SetPVarFloat(playerid, PVAR_TAG"oldZ", oldpos[2]);
	SetPVarFloat(playerid, PVAR_TAG"oldR", oldpos[3]);
	SetPVarInt(playerid, PVAR_TAG"oldInt", oldint);
	#endif
	
	if (spot == -1)
	{
		spot = 0;
		for (new p, mp = GetMaxPlayers(); p != mp; p++)
		{
			if (IsPlayerConnected(p) && !IsPlayerNPC(p))
			{
				new pRace = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
				if (pRace && pRace - 2 == race)
				{
					spot++;
				}
			}
		}
	}
	
	new Float: spos[3], Float: size[3], Float: offset[2], Float: angle = GetAngleToPos(raceCheckpointList[race][0][0], raceCheckpointList[race][0][1], raceCheckpointList[race][1][0], raceCheckpointList[race][1][1]), Float: temp;
	GetVehicleModelInfo(RACE_VEHICLE_MODEL, VEHICLE_MODEL_INFO_SIZE, size[0], size[1], size[2]);
	offset[0] = (spot & 1) ? (size[0] * 0.75) : -(size[0] * 0.75);
	offset[1] = -float((spot / 2) * floatround((size[1] * 1.2), floatround_ceil));
		
	// Thanks to Mauzen for offset calculation! ( http://forum.sa-mp.com/showthread.php?t=270427 )
	new Float: flSin = floatsin(-angle, degrees), Float: flCos = floatcos(-angle, degrees);
	spos[0] = flSin * offset[1] + flCos * offset[0] + raceCheckpointList[race][0][0];
	spos[1] = flCos * offset[1] - flSin * offset[0] + raceCheckpointList[race][0][1];
	GetNodePos(NearestNodeFromPoint(spos[0], spos[1], raceCheckpointList[race][0][2] + 1.5, 50.0, -1, 1), temp, temp, spos[2]);
	
	new veh = CreateVehicle(RACE_VEHICLE_MODEL, spos[0], spos[1], spos[2] + 2.5, angle, RACE_VEHICLE_COL1, RACE_VEHICLE_COL2, 0);
	SetPlayerInterior(playerid, 0);
	PutPlayerInVehicle(playerid, veh, 0);
	TogglePlayerControllable(playerid, false);
	SetPVarInt(playerid, PVAR_TAG"currentVehID", veh);
	SetPVarInt(playerid, PVAR_TAG"currentCPID", 1);
	return 1;
}

stock startRace(race)
{
	if (!race) return 1;

	SetTimerEx("countdownTimer", 1000, false, "ii", race, 3);
	return 1;
}

forward countdownTimer(race, count);
public countdownTimer(race, count)
{
	new str[2];
	str[0] = '0' + count;

	for (new p, mp = GetMaxPlayers(); p != mp; p++)
	{
		if (IsPlayerConnected(p) && !IsPlayerNPC(p))
		{
			new pRace = GetPVarInt(p, PVAR_TAG"currentRaceID");
			if (pRace && pRace - 2 == race)
			{
				switch (count)
				{
					case 0:
					{
						GameTextForPlayer(p, "GO!", 1250, 3);
						TogglePlayerControllable(p, true);
						PlayerPlaySound(p, 1057, 0.0, 0.0, 0.0);
						SendClientMessage(p, COL_TEXT_REG, " [!] NOTE: If you get stuck, you can respawn at the last checkpoint by using /respawninrace.");
					}
					case 1, 2:
					{
						GameTextForPlayer(p, str, 1250, 3);
						PlayerPlaySound(p, 1056, 0.0, 0.0, 0.0);
					}
					case 3:
					{
						
						GameTextForPlayer(p, str, 1250, 3);
						PlayerPlaySound(p, 1056, 0.0, 0.0, 0.0);
						pRace -= 2;
						setCheckpoint(p, pRace, 2);
					}
				}
			}
		}
	}
	if (count) 
	{
		SetTimerEx("countdownTimer", 1000, false, "ii", race, count - 1);
	}
	else
	{
		raceStarted{race} = true;
	}
	return 1;
}

public OnPlayerEnterRaceCheckpoint(playerid)
{
	new race = GetPVarInt(playerid, PVAR_TAG"currentRaceID");
	if (race)
	{
		race -= 2;
		new cp = GetPVarInt(playerid, PVAR_TAG"currentCPID") + 1;
		setCheckpoint(playerid, race, cp);
		PlayerPlaySound(playerid, 1058, 0.0, 0.0, 0.0);
	}	
	return 1;
}

stock setCheckpoint(playerid, race, cp)
{
	if (cp == MAX_CHECKPOINTS || (!raceCheckpointList[race][cp][0] && !raceCheckpointList[race][cp][1] && !raceCheckpointList[race][cp][2]))
	{
		DisablePlayerRaceCheckpoint(playerid);
		raceFinishedPeople{race}++;
		
		new nmb[3];
		switch (raceFinishedPeople{race})
		{
			case 1: GameTextForPlayer(playerid, "1st place!", 3500, 3), nmb = "st";
			case 2: GameTextForPlayer(playerid, "2nd place!", 3500, 3), nmb = "nd";
			case 3: GameTextForPlayer(playerid, "3rd place!", 3500, 3), nmb = "rd";
			default:
			{
				new str[12];
				format(str, sizeof(str), "%ith place!", raceFinishedPeople{race});
				GameTextForPlayer(playerid, str, 3500, 3);
				nmb = "th";
			}
		}
		new text[128], pName[MAX_PLAYER_NAME], oName[MAX_PLAYER_NAME], len;
		GetPlayerName(playerid, pName, MAX_PLAYER_NAME);
		len = GetPlayerName(raceOwners[race], oName, MAX_PLAYER_NAME);
		
		switch (oName[len - 1])
		{
			case 's', 'z': format(text, sizeof(text), " [!] FINISH: %s finished %i%s place in %s' race!", pName, raceFinishedPeople{race}, nmb, oName);
			default: format(text, sizeof(text), " [!] FINISH: %s finished %i%s place in %s's race!", pName, raceFinishedPeople{race}, nmb, oName);
		}
		SendClientMessageToAll(COL_TEXT_WIN, text);
		
		if (raceFinishedPeople{race} == 1)
		{
			raceEndTimers{race} = SetTimerEx("raceEndingTimer", 60000, false, "i", race);
			
			for (new p, mp = GetMaxPlayers(); p != mp; p++)
			{
				if (IsPlayerConnected(p) && !IsPlayerNPC(p))
				{
					new pRace = GetPVarInt(p, PVAR_TAG"currentRaceID");
					if (pRace && pRace - 2 == race)
					{
						SendClientMessage(p, COL_TEXT_REG, " [!] NOTE: The race will end in 60 seconds.");
					}
				}
			}
		}
		
		SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You can leave the race by using /leaverace or wait until the race ends.");
	}
	else if (cp + 1 < MAX_CHECKPOINTS && (raceCheckpointList[race][cp + 1][0] || raceCheckpointList[race][cp + 1][1] || raceCheckpointList[race][cp + 1][2]))
	{
		SetPlayerRaceCheckpoint(playerid, 0, raceCheckpointList[race][cp][0], raceCheckpointList[race][cp][1], raceCheckpointList[race][cp][2], raceCheckpointList[race][cp + 1][0], raceCheckpointList[race][cp + 1][1], raceCheckpointList[race][cp + 1][2], 15.0);
	}
	else
	{
		SetPlayerRaceCheckpoint(playerid, 1, raceCheckpointList[race][cp][0], raceCheckpointList[race][cp][1], raceCheckpointList[race][cp][2], 0.0, 0.0, 0.0, 15.0);
	}
	
	for (new ci, icon = 99; ci != 2; ci++, icon--)
	{
		new cin = cp + 1 + ci;
		if (cin < MAX_CHECKPOINTS && (raceCheckpointList[race][cin][0] || raceCheckpointList[race][cin][1] || raceCheckpointList[race][cin][2]))
		{
			SetPlayerMapIcon(playerid, icon, raceCheckpointList[race][cin][0], raceCheckpointList[race][cin][1], raceCheckpointList[race][cin][2], 0, COL_MAP_CP, 0);
		}
		else 
		{
			RemovePlayerMapIcon(playerid, icon);
		}
	}
	SetPVarInt(playerid, PVAR_TAG"currentCPID", cp);
	return 1;
}

forward raceEndingTimer(race);
public raceEndingTimer(race)
{
	for (new p, mp = GetMaxPlayers(); p != mp; p++)
	{
		if (IsPlayerConnected(p) && !IsPlayerNPC(p))
		{
			new pRace = GetPVarInt(p, PVAR_TAG"currentRaceID");
			if (pRace && pRace - 2 == race)
			{
				new cp = GetPVarInt(p, PVAR_TAG"currentCPID");
				if (cp == MAX_CHECKPOINTS || raceCheckpointList[race][cp][0] || raceCheckpointList[race][cp][1] || raceCheckpointList[race][cp][2])
				{
					SendClientMessage(p, COL_TEXT_REG, " [!] NOTE: Too slow! You didn't finished before the race ended.");
				}
				else
				{
					#if REMEMBER_OLD_POSITION == true
					SendClientMessage(p, COL_TEXT_REG, " [!] NOTE: The race has been ended. You have been respawned at your old position.");
					#else
					SendClientMessage(p, COL_TEXT_REG, " [!] NOTE: The race has been ended.");
					#endif
				}
			}
		}
	}
	cleanRace(race);
	return 1;
}

stock respawnPlayer(playerid, race)
{		
	new veh = GetPVarInt(playerid, PVAR_TAG"currentVehID"), cp = GetPVarInt(playerid, PVAR_TAG"currentCPID") - 1, 
		Float: angle = GetAngleToPos(raceCheckpointList[race][cp][0], raceCheckpointList[race][cp][1], raceCheckpointList[race][cp + 1][0], raceCheckpointList[race][cp + 1][1]);
	
	if (veh)
	{
		DestroyVehicle(veh);
	}
	
	veh = CreateVehicle(RACE_VEHICLE_MODEL, raceCheckpointList[race][cp][0], raceCheckpointList[race][cp][1], raceCheckpointList[race][cp][2] + 3.5, angle, RACE_VEHICLE_COL1, RACE_VEHICLE_COL2, 0);
	PutPlayerInVehicle(playerid, veh, 0);
	SetPVarInt(playerid, PVAR_TAG"currentVehID", veh);
	
	TogglePlayerControllable(playerid, false);
	SetPVarInt(playerid, PVAR_TAG"respawnTimer", SetTimerEx("respawnUnfreeze", 3000, false, "i", playerid));
	GameTextForPlayer(playerid, "Respawning", 3250, 3);
	SendClientMessage(playerid, COL_TEXT_REG, " [!] NOTE: You have been respawned in the race.");
	return 1;
}

forward respawnUnfreeze(playerid);
public respawnUnfreeze(playerid)
{
	if (!IsPlayerConnected(playerid)) return 1;
	TogglePlayerControllable(playerid, true);
	return 1;
}

stock getFirstEmptyCPSlot(race)
{
	if (0 <= race < MAX_RACES && raceOwners[race] != INVALID_PLAYER_ID)
	{
		for (new c; c != MAX_CHECKPOINTS; c++)
		{
			if (raceCheckpointList[race][c][0] || raceCheckpointList[race][c][1] || raceCheckpointList[race][c][2]) continue;
			return c;
		}
	}
	return MAX_CHECKPOINTS;
}